<!DOCTYPE html>
<html lang="de">
<head>
   <meta charset="utf-8">
   <title>Kodieren</title>
</head>

<!-- Allgemeine Infos:

     Es ist wichtig, dass die Bilddatei in einem verlustfreien Pixelformat vorliegt, hier im PNG-Format.
     Jedem Pixel (Bildpunkt) einer solchen Datei ist eine bestimmte Farbe zugeordnet.
     Diese Farbe setzt sich aus drei Anteilen zusammen: Rot, Grün und Blau.
     Jeder Farbanteil wird in einem Byte, also in acht Bit kodiert.
     Damit ergeben sich Werte zwischen 0 (geringer Anteil dieser Farbe) und 255 (hoher Anteil dieser Farbe).

     Nehmen wir für einen bestimmten Bildpunkt einen Rot-Wert von 83 an. Als Dualzahl ist das die Bitfolge 0101 0011.
     Das Auge nimmt den Unterschied zwischen den Farbwerten im Bereich von 80 (0101 0000) bis 87 (0101 0111) nicht wahr.
     Daher können wir die drei niederwertigen Bit jedes Anteils einer Farbe (die drei Bit ganz rechts) verändern, ohne dass es bemerkt wird.
     Es gibt drei Farbanteile pro Pixel, das ergibt insgesamt 9 Bit: genügend Platz, um jeweils ein Byte der Lastdatei zu verstecken.

     Bei den beiden Programmen zum Verstecken und zum Rekonstruieren der Daten wird mit einzelnen Bytes gearbeitet.
     Daher kommen die Standardfunktionen strlen(), ord() und chr() statt der Multibyte-Funktionen
        mb_strlen(), mb_ord() und mb_chr() zum Einsatz, mit gleicher Wirkungsweise. -->

<body>
<?php
   /* Mithilfe dieses Programms in der Datei kodieren.php wird die Lastdatei innerhalb der Bilddatei versteckt. 
      Die Namen der beiden Dateien werden in den beiden Variablen $bildDatei und $lastDatei gespeichert. */
   $bildDatei = "bild.png";
   $lastDatei = "beispiel.pdf";

   /* Die Funktion getimagesize() liefert ein Feld mit Informationen über eine Bilddatei.
      Die beiden Feldelemente 0 und 1 beinhalten die Breite und die Höhe des Bilds in Pixeln.
      Diese Informationen werden später für den Zugriff auf die einzelnen Pixel des Bilds benötigt. */
   $bildInfo = getimagesize($bildDatei);
   $bildBreite = $bildInfo[0]; 
   $bildHoehe = $bildInfo[1]; 
   
   /* Die Funktion file_get_contents() liest sämtliche Bytes einer Datei in eine Zeichenkette.
      Mithilfe der Funktion strlen() wird die Länge der Zeichenkette festgestellt, hier also die Anzahl der zu versteckenden Bytes.
      Zu dieser Anzahl werden noch vier Bytes hinzugerechnet. In diesen vier Bytes wird die Größe der Lastdatei als Zahl gespeichert.
      Ist die auf diese Weise ermittelte Gesamtanzahl der benötigten Bytes zu groß für die Bilddatei, wird das Programm beendet. */
   $lastString = file_get_contents($lastDatei);
   $lastGroesse = strlen($lastString);
   if($lastGroesse > $bildBreite * $bildHoehe - 4)
      exit("Last zu groß");

   /* Die Größe der Lastdatei wird kodiert und in einem Feld gespeichert.
      Dieses besteht aus vier einzelnen Bytes, die später in den ersten vier Bytes der Bilddatei versteckt werden.
      Daraus kann das dekodierende Programm ermitteln, wie viele der nachfolgenden Bytes zur Lastdatei gehören.

      Hier hat die Lastdatei beispiel.pdf eine Größe von 360.226 Byte.
      Schreiben wir diesen Wert als Dualzahl mit 32 Stellen: 0000 0000 0000 0101 0111 1111 0010 0010
      Das sind vier Gruppen zu jeweils acht Bit. Jede Gruppe à acht Bit wird einzeln mithilfe von Bit-Operatoren ermittelt.
      Dabei werden die Bit-Operatoren >> zum Verschieben von Bits und & (bitweises logisches Und) zum Filtern von Bits genutzt.

      Ein Beispiel für die Ermittlung des Elements 2 des Felds:
      Nach dem Ausdruck $lastGroesse >> 8 sind die Bits der Dualzahl um acht Bit nach rechts verschoben: 0000 0000 0000 0000 0000 0101 0111 1111
      Mit dem Präfix 0x wird eine Hexadezimalzahl eingeleitet. Die Zahl 0xFF entspricht der Dualzahl 1111 1111.
      Nach dem Ausdruck ($lastGroesse >> 8) & 0xFF verbleiben nur noch die acht Bits ganz rechts: 0000 0000 0000 0000 0000 0000 0111 1111 */
   $lastGroesseFeld[0] = ($lastGroesse >> 24) & 0xFF;
   $lastGroesseFeld[1] = ($lastGroesse >> 16) & 0xFF;
   $lastGroesseFeld[2] = ($lastGroesse >> 8) & 0xFF;
   $lastGroesseFeld[3] = $lastGroesse & 0xFF;

   /* Der Inhalt der Bilddatei wird in einer Zeichenkette gespeichert.
      Aus dieser Zeichenkette wird ein Image-Objekt erzeugt. */
   $img = imagecreatefromstring(file_get_contents($bildDatei));

   /* Mithilfe einer doppelten Schleife wird auf die einzelnen Pixel des Bilds zugegriffen.
      Der Index des Bytes, das im jeweiligen Pixel versteckt werden soll, wird mithilfe der beiden Schleifenvariablen ermittelt. */
   for($x=0; $x<$bildBreite; $x++)
   {
      for($y=0; $y<$bildHoehe; $y++)
      {
         /* Die folgenden Operationen werden nur durchgeführt, falls der Index kleiner ist als die Summe
               aus der Größe der Lastdatei und der Anzahl der Bytes des Felds. */
         $index = $x * $bildHoehe + $y; 

         if($index < $lastGroesse + 4)
         {
            /* Hat der Index einen Wert von 0 bis 3, wird ein Byte aus dem Feld,
                  in dem die Größe der Lastdatei steht, in der Variablen $zahl gespeichert. */
            if($index < 4)
               $zahl = $lastGroesseFeld[$index];
            /* Hat der Index einen Mindestwert von 4, wird mithilfe der Funktion ord() der ASCII-Wert eines Bytes aus der Lastdatei
               in der Variablen $zahl gespeichert. Dabei ist eine Verschiebung um 4 Bytes zu berücksichtigen, wegen der vorher
               gespeicherten Größe der Lastdatei. Hat also zum Beispiel der Index den Wert 14, wird auf Byte 10 der Lastdatei zugegriffen. */
            else
               $zahl = ord($lastString[$index-4]);

            /* Das Byte in der Variablen $zahl wird in einzelne Bits zerlegt.
               Die ersten zwei Bits sollen in den letzten zwei Bits des Rot-Anteils des Pixels gespeichert werden.
               Die nächsten drei Bits sollen in den letzten drei Bits des Grün-Anteils gespeichert werden.
               Die letzten drei Bits sollen in den letzten 3 Bits des Blau-Anteils gespeichert werden.
               Dazu werden drei kombinierte Bit-Operationen vorgenommen. */

            /* Mithilfe der Hexadezimalzahl 0xC0 werden die ersten zwei Bits herausgefiltert. Sie werden um sechs Stellen nach
               ganz rechts verschoben und anschließend mithilfe des Bit-Operators << um zwei Bytes nach links verschoben. */
            $rot = (($zahl & 0xC0) >> 6) << 16;

            /* Mithilfe der Hexadezimalzahl 0x38 werden die nächsten drei Bits herausgefiltert. Sie werden um drei Stellen nach
               ganz rechts verschoben und anschließend um ein Byte nach links verschoben. */
            $gruen = (($zahl & 0x38) >> 3) << 8;

            /* Mithilfe der Hexadezimalzahl 0x07 werden die letzten drei Bits herausgefiltert. Sie müssen nicht mehr verschoben werden. */
            $blau = $zahl & 0x07;

            /* Der Bit-Operator | (bitweises logisches Oder) dient zur Verknüpfung dieser drei Bytes.
               Sie werden zu einer Farbe zusammengesetzt. */
            $lastFarbe = $rot | $gruen | $blau;

            /* Mithilfe der Funktion imagecolorat() wird die Farbkombination eines Pixels im Bild ermittelt. */
            $bildFarbe = imagecolorat($img, $x, $y);

            /* Die hexadezimale Zahl 0xFCF8F8 entspricht der Dualzahl 1111 1100 1111 1000 1111 1000.
               Nach dem Ausdruck $bildFarbe & 0xFCF8F8 bleiben nur noch die Werte derjenigen Bits erhalten,
                  die nicht von der Lastinformation überschrieben werden sollen.
               Mithilfe des Ausdrucks ($bildFarbe & 0xFCF8F8) | $lastFarbe werden die Bits der Originalfarbe
                  mit den Bits der Lastinformation verknüpft. */
            $bildFarbe = ($bildFarbe & 0xFCF8F8) | $lastFarbe;

            /* Die Funktion imagesetpixel() besetzt das zuvor ermittelte Pixel mit der neuen Farbkombination. */
            imagesetpixel($img, $x, $y, $bildFarbe);
         }
      }
   }

   /* Aus dem Namen der unveränderten Bilddatei wird der Name der veränderten Bilddatei gebildet. Aus bild.png wird also bild_neu.png.
      Das Image-Objekt wird an die neue Bilddatei übergeben und anschließend zerstört. */
   $bildDateiFeld = mb_split("\.", $bildDatei);
   imagepng($img, $bildDateiFeld[0] . "_neu." . $bildDateiFeld[1]);
   imagedestroy($img);
?>
</body>
</html>
